// Popup controller with auth
const API_BASE = 'https://auditmysite.app';
let crawler = null;
let authToken = null;

const $ = (sel) => document.querySelector(sel);
const $$ = (sel) => document.querySelectorAll(sel);

// Elements
const authSection = $('#auth-section');
const noLicenseSection = $('#no-license-section');
const startSection = $('#start-section');
const crawlSection = $('#crawl-section');
const urlInput = $('#url-input');
const startBtn = $('#start-btn');
const stopBtn = $('#stop-btn');
const exportBtn = $('#export-btn');
const newBtn = $('#new-btn');
const progressBar = $('#progress-bar');
const progressText = $('#progress-text');
const authBtn = $('#auth-btn');
const authEmail = $('#auth-email');
const authPassword = $('#auth-password');
const authError = $('#auth-error');
const userBar = $('#user-bar');
const userEmailEl = $('#user-email');
const logoutBtn = $('#logout-btn');

// Check stored auth on load
chrome.storage.local.get(['ams_token', 'ams_email'], (data) => {
  if (data.ams_token) {
    validateToken(data.ams_token, data.ams_email);
  }
  // Auto-fill current tab URL
  chrome.tabs?.query({ active: true, currentWindow: true }, (tabs) => {
    if (tabs[0]?.url && tabs[0].url.startsWith('http')) {
      try { urlInput.value = new URL(tabs[0].url).origin; } catch {}
    }
  });
});

// Validate stored token
async function validateToken(token, email) {
  try {
    const res = await fetch(`${API_BASE}/api/auth/extension?token=${encodeURIComponent(token)}`);
    const data = await res.json();
    if (data.valid && data.licensed) {
      authToken = token;
      showCrawler(data.email || email);
    } else if (data.valid && !data.licensed) {
      showNoLicense(data.email || email);
    } else {
      // Token expired
      chrome.storage.local.remove(['ams_token', 'ams_email']);
    }
  } catch {
    // Network error — allow offline use if token exists
    if (token) {
      authToken = token;
      showCrawler(email || 'offline');
    }
  }
}

function showCrawler(email) {
  authSection.classList.add('hidden');
  noLicenseSection.classList.add('hidden');
  startSection.classList.remove('hidden');
  userBar.classList.remove('hidden');
  userEmailEl.textContent = email;
}

function showNoLicense(email) {
  authSection.classList.add('hidden');
  noLicenseSection.classList.remove('hidden');
  startSection.classList.add('hidden');
  userBar.classList.remove('hidden');
  userEmailEl.textContent = email;
}

// Login
authBtn.addEventListener('click', async () => {
  const email = authEmail.value.trim();
  const password = authPassword.value;
  if (!email || !password) return;

  authBtn.disabled = true;
  authBtn.textContent = 'Signing in...';
  authError.classList.add('hidden');

  try {
    const res = await fetch(`${API_BASE}/api/auth/extension`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ email, password }),
    });
    const data = await res.json();

    if (!res.ok) {
      authError.textContent = data.error || 'Login failed';
      authError.classList.remove('hidden');
      return;
    }

    // Store token
    chrome.storage.local.set({ ams_token: data.token, ams_email: data.email });

    if (data.licensed) {
      authToken = data.token;
      showCrawler(data.email);
    } else {
      showNoLicense(data.email);
    }
  } catch (err) {
    authError.textContent = 'Network error. Check your connection.';
    authError.classList.remove('hidden');
  } finally {
    authBtn.disabled = false;
    authBtn.textContent = 'Sign In';
  }
});

// Enter key on password field
authPassword.addEventListener('keydown', (e) => {
  if (e.key === 'Enter') authBtn.click();
});

// Logout
logoutBtn.addEventListener('click', () => {
  chrome.storage.local.remove(['ams_token', 'ams_email']);
  authToken = null;
  authSection.classList.remove('hidden');
  startSection.classList.add('hidden');
  crawlSection.classList.add('hidden');
  noLicenseSection.classList.add('hidden');
  userBar.classList.add('hidden');
  authEmail.value = '';
  authPassword.value = '';
});

// Tab switching
$$('.tab').forEach(tab => {
  tab.addEventListener('click', () => {
    $$('.tab').forEach(t => t.classList.remove('active'));
    tab.classList.add('active');
    const target = tab.dataset.tab;
    $('#tab-issues').style.display = target === 'issues' ? '' : 'none';
    $('#tab-pages').style.display = target === 'pages' ? '' : 'none';
  });
});

// Start crawl — unlimited pages
startBtn.addEventListener('click', () => {
  let url = urlInput.value.trim();
  if (!url) return;
  if (!url.startsWith('http')) url = 'https://' + url;

  try { new URL(url); } catch { return alert('Invalid URL'); }

  startSection.classList.add('hidden');
  crawlSection.classList.remove('hidden');

  // No page limit — pass Infinity for unlimited
  crawler = new SEOCrawler(url, Infinity, updateUI);
  crawler.start().then(() => {
    progressText.textContent = `Crawl complete — ${crawler.pages.length} pages analyzed`;
    exportBtn.disabled = false;
    syncBtn.disabled = false;
  });
});

// Stop
stopBtn.addEventListener('click', () => {
  if (crawler) {
    crawler.stop();
    progressText.textContent = `Crawl stopped — ${crawler.pages.length} pages analyzed`;
    exportBtn.disabled = false;
    if (crawler.pages.length > 0) syncBtn.disabled = false;
  }
});

// New crawl
newBtn.addEventListener('click', () => {
  crawler = null;
  crawlSection.classList.add('hidden');
  startSection.classList.remove('hidden');
  $('#tab-issues').innerHTML = '';
  $('#tab-pages').innerHTML = '';
  exportBtn.disabled = true;
  syncBtn.disabled = true;
  syncBtn.innerHTML = '<svg style="width:12px;height:12px;display:inline-block;vertical-align:middle;margin-right:4px" fill="none" viewBox="0 0 24 24" stroke="currentColor" stroke-width="2"><path stroke-linecap="round" stroke-linejoin="round" d="M7 16V4m0 0L3 8m4-4l4 4m6 0v12m0 0l4-4m-4 4l-4-4"/></svg>Sync to Dashboard';
  syncStatus.classList.add('hidden');
});

// Sync to Dashboard
const syncBtn = $('#sync-btn');
const syncStatus = $('#sync-status');

syncBtn.addEventListener('click', async () => {
  if (!crawler || !authToken) return;

  // Check if user is logged in
  if (!authToken) {
    syncStatus.classList.remove('hidden');
    syncStatus.style.background = '#eab30820';
    syncStatus.style.color = '#eab308';
    syncStatus.innerHTML = 'Sign in to sync results. <a href="https://auditmysite.app/extension" target="_blank" style="color:#3b82f6;text-decoration:none">Upgrade to sync results</a>';
    return;
  }

  syncBtn.disabled = true;
  syncBtn.textContent = 'Syncing...';
  syncStatus.classList.remove('hidden');
  syncStatus.style.background = '#3b82f620';
  syncStatus.style.color = '#3b82f6';
  syncStatus.textContent = 'Syncing to dashboard...';

  try {
    const domain = new URL(crawler.startUrl).hostname.replace(/^www\./, '');
    const pageData = crawler.pages.map(p => ({
      url: p.url,
      title: p.title || '',
      meta_description: p.metaDesc || '',
      h1: p.h1 || '',
      issues: (p.issues || []).map(i => i.type || i.message || String(i)),
      links: [], // internal links collected during crawl
    }));

    const res = await fetch(`${API_BASE}/api/optimize/extension-sync`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ token: authToken, domain, pages: pageData }),
    });

    const data = await res.json();

    if (res.ok && data.success) {
      syncStatus.style.background = '#22c55e20';
      syncStatus.style.color = '#22c55e';
      syncStatus.innerHTML = `Synced ${data.pages_synced} pages! <a href="${API_BASE}/dashboard/${data.site_id}" target="_blank" style="color:#3b82f6;text-decoration:none;font-weight:600">View in Dashboard &rarr;</a>`;
      syncBtn.textContent = 'Synced';
    } else {
      syncStatus.style.background = '#ef444420';
      syncStatus.style.color = '#ef4444';
      syncStatus.textContent = data.error || 'Sync failed. Try again.';
      syncBtn.disabled = false;
      syncBtn.innerHTML = '<svg style="width:12px;height:12px;display:inline-block;vertical-align:middle;margin-right:4px" fill="none" viewBox="0 0 24 24" stroke="currentColor" stroke-width="2"><path stroke-linecap="round" stroke-linejoin="round" d="M7 16V4m0 0L3 8m4-4l4 4m6 0v12m0 0l4-4m-4 4l-4-4"/></svg>Sync to Dashboard';
    }
  } catch (err) {
    syncStatus.style.background = '#ef444420';
    syncStatus.style.color = '#ef4444';
    syncStatus.textContent = 'Network error. Check your connection.';
    syncBtn.disabled = false;
    syncBtn.innerHTML = '<svg style="width:12px;height:12px;display:inline-block;vertical-align:middle;margin-right:4px" fill="none" viewBox="0 0 24 24" stroke="currentColor" stroke-width="2"><path stroke-linecap="round" stroke-linejoin="round" d="M7 16V4m0 0L3 8m4-4l4 4m6 0v12m0 0l4-4m-4 4l-4-4"/></svg>Sync to Dashboard';
  }
});

// Export CSV
exportBtn.addEventListener('click', () => {
  if (!crawler) return;
  const csv = crawler.toCSV();
  const blob = new Blob([csv], { type: 'text/csv' });
  const url = URL.createObjectURL(blob);
  const a = document.createElement('a');
  a.href = url;
  a.download = `audit-${new URL(crawler.startUrl).hostname}-${Date.now()}.csv`;
  a.click();
  URL.revokeObjectURL(url);
});

function updateUI(data) {
  $('#s-crawled').textContent = data.crawled;
  $('#s-ok').textContent = data.ok;
  $('#s-warnings').textContent = data.warnings;
  $('#s-errors').textContent = data.errors;

  const pct = data.maxPages === Infinity ? 0 : Math.round((data.crawled / data.maxPages) * 100);
  progressBar.style.width = data.maxPages === Infinity ? '100%' : `${Math.min(pct, 100)}%`;
  progressText.textContent = `Crawling... ${data.crawled} pages (${data.queued} in queue)`;

  // Issues grouped by type
  const grouped = {};
  for (const issue of data.issues) {
    if (!grouped[issue.type]) grouped[issue.type] = { type: issue.type, severity: issue.severity, urls: [] };
    grouped[issue.type].urls.push(issue.url);
  }
  const sorted = Object.values(grouped).sort((a, b) => {
    const sev = { high: 0, medium: 1, low: 2 };
    return (sev[a.severity] || 2) - (sev[b.severity] || 2);
  });

  const issuesEl = $('#tab-issues');
  issuesEl.innerHTML = sorted.length === 0 ? '<div class="empty">No issues found yet...</div>' : '';
  for (const group of sorted) {
    const div = document.createElement('div');
    div.className = 'issue';
    div.innerHTML = `
      <div class="issue-header">
        <span class="issue-type"><span class="issue-severity severity-${group.severity}"></span>${group.type}</span>
        <span class="issue-count">${group.urls.length}</span>
      </div>
      <div class="issue-url">${group.urls.slice(0, 3).map(u => truncate(u, 60)).join('<br>')}${group.urls.length > 3 ? `<br>...and ${group.urls.length - 3} more` : ''}</div>
    `;
    issuesEl.appendChild(div);
  }

  // Pages list (show last 50)
  const pagesEl = $('#tab-pages');
  pagesEl.innerHTML = '';
  for (const page of data.pages.slice(-50)) {
    const div = document.createElement('div');
    div.className = 'page-row';
    const color = page.issueCount === 0 ? '#22c55e' : page.issueCount <= 2 ? '#eab308' : '#ef4444';
    div.innerHTML = `
      <div class="page-status" style="background:${color}"></div>
      <div class="page-url" title="${page.url}">${truncate(page.url.replace(/^https?:\/\//, ''), 45)}</div>
      <div class="page-issues">${page.status} · ${page.issueCount} issues</div>
    `;
    pagesEl.appendChild(div);
  }
}

function truncate(str, max) {
  return str.length > max ? str.substring(0, max) + '...' : str;
}
